/*+ MetaParser.cpp
 *
 ******************************************************************************
 *
 *                        Trimble Navigation Limited
 *                           645 North Mary Avenue
 *                              P.O. Box 3642
 *                         Sunnyvale, CA 94088-3642
 *
 ******************************************************************************
 *
 *    Copyright  2005 Trimble Navigation Ltd.
 *    All Rights Reserved
 *
 ******************************************************************************
 *
 * Description:
 *    This file implements the CMetaParser class.
 *            
 * Revision History:
 *    01-11-2007    Jacob Norda
 *                  Written
 *
 * Notes:
 *
-*/

/*---------------------------------------------------------------------------*\
 |                         I N C L U D E   F I L E S
\*---------------------------------------------------------------------------*/
#include "stdafx.h"
#include "MetaParser.h"
#include "TsipParser.h"
#include "NmeaParser.h"
#include "CommPort.h"
#include <math.h>


/*---------------------------------------------------------------------------*\
 |         P R O T O C O L   P R O C E S S O R   R O U T I N E S 
\*---------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------
Function:       ReceiveAndParsePkt

Description:    Receives a complete packet from a specified serial port and 
				returns the ASCII-representation of it. The packet is formatted
				into the supplied parameters for raw output/logging purposes. 

                NOTE: This function returns as soon as a valid packet has been 
                received on the specified serial port or an error has been 
				detected. It will block the calling thread until then.

Parameters:     ptSerialPort - a pointer to the serial port object from which
                               data can be received.
                ucPkt        - a memory buffer where the entire packet 
                               will be stored.
                pPktLen      - a pointer to a variable to be updated with the
                               packet size (which includes the packet header
                               and trailing bytes).

Return Value:   None
-----------------------------------------------------------------------------*/
CString CMetaParser::ReceiveAndParsePkt (CCommPort *ptSerialPort, 
                              U8 ucPkt[], 
                              int *pPktLen)
{
    U8 ucByte;
    int nParseState;
	int nProtocolState = PROTOCOL_DETERMINE; 
	int nTimeout = MAX_PKT_LEN*3; 

    // This function runs in a permanent loop until a valid packet has
    // been received on the specified serial port or an error has been 
	// detected.
    while (nTimeout--)
    {
		// Read one byte at a time
		ptSerialPort->Read(&ucByte);

		switch(nProtocolState) 
		{
		default: 
		case PROTOCOL_DETERMINE: 
			if (m_TsipParser.IsStartOfTsip(ucByte)) 
			{ 
				nProtocolState = PROTOCOL_TSIP;
				m_TsipParser.Reset(); 
				nParseState = m_TsipParser.ReceiveByte(ucByte, ucPkt, pPktLen);
			} 
			else if (m_NmeaParser.IsStartOfNmea(ucByte)) 
			{ 
				nProtocolState = PROTOCOL_NMEA; 
				m_NmeaParser.Reset(); 
				nParseState = m_NmeaParser.ReceiveByte(ucByte, ucPkt, pPktLen);
			}
			break;
		case PROTOCOL_TSIP: 
			nParseState = m_TsipParser.ReceiveByte(ucByte, ucPkt, pPktLen);
			break;
		case PROTOCOL_NMEA:
			nParseState = m_NmeaParser.ReceiveByte(ucByte, ucPkt, pPktLen);
			break;
		}

		if (nProtocolState != PROTOCOL_DETERMINE) switch (nParseState) 
		{
		case MSG_IN_COMPLETE: 
			switch(nProtocolState) 
			{
			case PROTOCOL_TSIP: 
				return "TSIP> " + m_TsipParser.ParsePkt(ucPkt, *pPktLen);
			case PROTOCOL_NMEA:
				return "NMEA> " + m_NmeaParser.ParsePkt(ucPkt, *pPktLen);
			}
			return "ERROR> Undefined state"; 
			
		case MSG_IN_ERROR: 
			return "ERROR> Received bad packet"; 
		}
    }
	return "ERROR> Timeout"; 
}


/*-----------------------------------------------------------------------------
Function:       SendNmeaPkt

Description:    This function sends a predetermined sample packet (VR) to the 
				supplied serial port. 

Parameters:     pPort   - The serial port to send the sample packet to.

Return Value:   None.
-----------------------------------------------------------------------------*/
void CMetaParser::SendNmeaPkt(CCommPort* pPort)
{
	m_NmeaParser.SendPkt(pPort); 
}


